
/**
 ******************************************************************************
 *
 * @file        MG32x02z_USBD_HID_API.c
 * @brief       The USBD HID Code's c file.
 *
 * @par         Project
 *              MG32x02z
 * @version     V1.00
 * @date        2022/11/22
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/ 
#include "MG32x02z.h"
#include "MG32x02z_USBD_API.h"
#include "MG32x02z_USBD_EasyHID_API.h"


/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
EasyHID_TypeDef    eHID_EP;


/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/                                     
/* External variables --------------------------------------------------------*/



/**
 *******************************************************************************
 * @brief    HID relationship parameter initial.
 * @details    
 * @param    	
 * @return      
 *******************************************************************************
 */
void API_HID_Init(void)
{
    eHID_EP.RxFlag     = 0;
    eHID_EP.TxBusyFlag = 0;
    eHID_EP.TxLength   = 0;
}    

/**
 *******************************************************************************
 * @brief     To trigger HID endpoint transmit.
 * @details 
 * @param[in] Upstream_Buf:	 Pointer of transmit data source.
 * @param[in] Upstream_Length : Transmit data size.
 * @return    Return the trigger whether succeed or not.
 *   \n       HID_FAILURE: Trigger failure.
 *   \n       HID_SUCCESS: Trigger success.
 *******************************************************************************
 */
uint8_t API_HID_UpstreamData(uint8_t *Upstream_Buf, uint8_t Upstream_Length)
{
    uint8_t HID_USDataBLen;
    
    /*USB enumeration not yet success.*/
    if( (Ep0.USBStatus & USB_STATUS_EMULATION_MASK) != USB_STATUS_EMULATION_OK)
    {
        return(HID_FAILURE);
    }
    /*Previous data not yet transmit complete.*/
    if(eHID_EP.TxBusyFlag==1)
    {
        return(HID_FAILURE);
    }
    /*Handler upstream transmit.*/
    eHID_EP.TxBusyFlag = 1;
    
    eHID_EP.TxBuf    = Upstream_Buf;
    eHID_EP.TxLength = Upstream_Length;
    
    /*If upstream data bytes more than HID endpoint package size*/
    if( eHID_EP.TxLength > HID_DATAUPSTREAM_DATASIZE)
    {
        HID_USDataBLen = HID_DATAUPSTREAM_DATASIZE;
    }
    else
    {
        HID_USDataBLen = eHID_EP.TxLength;
    }
    
    eHID_EP.TxLength = eHID_EP.TxLength - HID_USDataBLen;
    
    /*Write upstream data to USB SRAM and telling USB hardware upstream data ready.*/
    USB_SetEndpointTXData( HID_DATAUPSTREAM_EP, eHID_EP.TxBuf,HID_USDataBLen);
    
    /*Shift stream buffer pointer.*/
    eHID_EP.TxBuf = eHID_EP.TxBuf + HID_USDataBLen;
    
    
    return(HID_SUCCESS);
}

/**
 *******************************************************************************
 * @brief    HID upstream endpoint IRQ handler.
 * @details    
 * @param    	
 * @return      
 *******************************************************************************
 */
void API_HID_UpstreamDataIRQHandler(void)
{
    uint8_t HID_USDataIRQBLen;
    
    /*Upstream data transmit complete.*/
    if(eHID_EP.TxLength==0)
    {
        eHID_EP.TxBusyFlag = 0;

        return;
    }
    /*If upstream data remaining data more than HID endpoint package size*/
    if( eHID_EP.TxLength > HID_DATAUPSTREAM_DATASIZE)
    {
        HID_USDataIRQBLen = HID_DATAUPSTREAM_DATASIZE;
    }
    else
    {
        HID_USDataIRQBLen = eHID_EP.TxLength;
    }
    
    eHID_EP.TxLength = eHID_EP.TxLength - HID_USDataIRQBLen;
    
    /*Write upstream data to USB SRAM and telling USB hardware upstream data ready.*/
    USB_SetEndpointTXData( HID_DATAUPSTREAM_EP, eHID_EP.TxBuf,HID_USDataIRQBLen);
    
    /*Shift stream buffer pointer.*/
    eHID_EP.TxBuf = eHID_EP.TxBuf + HID_USDataIRQBLen;
}

/**
 *******************************************************************************
 * @brief     Receive HID downstream endpoint data. 
 * @details    
 * @param[in] Downstream_Buf   : Downstream data store destination.	
 * @param[in] Downstream_Length: Downstream data size store destination.
 * @return    Return the trigger whether succeed or not.
 *   \n       HID_FAILURE: No downstream data.
 *   \n       HID_SUCCESS: Receive downstream data successfully.     
 *******************************************************************************
 */
uint8_t API_HID_DownstreamData(uint8_t *Downstream_Buf, uint8_t *Downstream_Length)
{
    /*USB enumeration not yet success.*/
    if( (Ep0.USBStatus & USB_STATUS_EMULATION_MASK) != USB_STATUS_EMULATION_OK)
    {
        return(HID_FAILURE);
    }
    
    /*Downstream no data*/
    if( eHID_EP.RxFlag == 0)
    {
        return(HID_FAILURE);
    }
    
    eHID_EP.RxFlag = 0;      
    
    /*Receive downstream data to *Downstream_Buf and release downstream endpoint.*/
    *Downstream_Length = (uint8_t)(USB_GetEndpointRXData(HID_DATADOWNSTREAM_EP,Downstream_Buf,ENABLE));
    
    return(HID_SUCCESS);
}   

/**
 *******************************************************************************
 * @brief    HID relationship parameter initial.
 * @details    
 * @param    	
 * @return      
 *******************************************************************************
 */
void API_HID_DownstreamDataIRQHandler(void)
{
    eHID_EP.RxFlag = 1;
} 













